import uuid
from django.db import models
from enum import Enum
from lib.choices import choices_the_options
from subscriptions.models import Profile, SalesPersons


class CollectionPlace(Enum):
    KOZHIKODE = 'kozhikode'
    KASARAGOD = 'kasaragod'
    DUBAI = 'dubai'
    QATAR = 'qatar'

class CollectionData(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    profiles = models.ForeignKey(Profile, on_delete=models.PROTECT, related_name='collections')
    company_name = models.CharField(max_length=200, null=True, blank=True)
    client_name = models.CharField(max_length=30, null=True, blank=True)
    way_of_contact = models.CharField(max_length=400, null=True, blank=True)
    types_of_business = models.CharField(max_length=200, null=True, blank=True)
    place = models.CharField(
        max_length=20, 
        choices=choices_the_options(CollectionPlace), 
        default=CollectionPlace.KASARAGOD.value
    )
    contact_no = models.CharField(max_length=16, null=True, blank=True)
    website = models.TextField(blank=True, null=True)
    social_media = models.TextField(blank=True, null=True)
    gmb = models.TextField(blank=True, null=True)
    interested = models.BooleanField(default=False)
    assigned_to = models.ForeignKey(
        SalesPersons, on_delete=models.SET_NULL, null=True, blank=True, related_name='collection_assigned_to'
    )
    address = models.TextField(blank=True, null=True)
    city = models.CharField(max_length=255, null=True, blank=True)
    state = models.CharField(max_length=255, null=True, blank=True)
    zip_code = models.CharField(max_length=255, null=True, blank=True)
    country = models.CharField(max_length=255, null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    created_by = models.ForeignKey(Profile, on_delete=models.PROTECT, related_name='created_collections')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.company_name or 'No Company'} - {self.client_name or 'No Client'}"

    class Meta:
        ordering = ['-created_at']
    
    def convert_to_lead(self, assigned_to=None):
        from .lead import Lead  
        return Lead.objects.create(
            company=self.company_name,
            first_name=self.client_name.split()[0] if self.client_name else '',
            last_name=' '.join(self.client_name.split()[1:]) if self.client_name else '',
            phone=self.contact_no,
            email=None,  
            lead_source='collection',
            collections=self,
            assigned_to=assigned_to or self.assigned_to,
            created_by=self.created_by,
            address=self.address,
            city=self.city,
            state=self.state,
            zip_code=self.zip_code,
            country=self.country
        )
